/** @file   animation.h
 * @brief   Declaration of Animation - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:37 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_ANIMATION_H
#define H_ENG2D_ANIMATION_H

#include <vector>
#include <string>
#include <fstream>
#include "sound.h"
#include "eng2d_dll.h"
#include "animframe.h"
#include "eng2d_Exception.h"


namespace eng2d {

/** @class  Animation
 * @brief   Every animation is an instance of this class.
 * @author  Tomi Lamminsaari
 *
 * Each Animation can have any number of frames and each frame can have
 * its own duration. The Animation - objects does hold any graphics data
 * so handling them is quite efficient.
 *
 * The usual way to handle animations is that you have a slt-vector where
 * you've pushed pointers to the bitmaps for this Animation. The Animation
 * itself refers to this vector by telling that from which index you
 * find the graphics for this frame.
 *
 * The AnimFrame instances can have sound information. This information
 * tells that which sample should be played when that frame is shown. The
 * Animation - class uses the services from @c Sound - class to produce the
 * sound.
 *
 * @see     AnimFrame
 */
class DLLIMPORT Animation
{
public:

  ///
  /// Static members and methods
  /// ==========================
  
  /** Assigns the pointers to the bitmaps that has the frame graphics
   * to the AnimFrame-objects the animation 'pA' has. If the frames of
   * animation 'pA' references to the framegraphics with bigger index
   * than there are elements in vector 'pT', this method will crash.
   * @param     pA                Pointer to the animation.
   * @param     pT                Pointer to the vector that has pointers
   *                              to the frame graphics.
   */
  static void assignGfx( Animation* pA, const std::vector<BITMAP*>* pT );
  
  /** As method 'assignRLE' but the framegraphics are RLE_SPRITEs instead
   * of normal bitmaps.
   * @param     pA                Pointer to the animation.
   * @param     pT                Pointer to the vector that has pointers
   *                              to the frame graphics.
   */
  static void assignRLE( Animation* pA, const std::vector<RLE_SPRITE*>* pT );
  
  /** Possible playmodes for the animation.
   */
  enum PlayMode {
    /** The animation will be played once. */
    MODE_ONCE,
    /** The animation will be loopped. */
    MODE_LOOP,
    /** The animation will be played back and forth. */
    MODE_PINGPONG
  };
  
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructs new empty animation.
   */
  Animation();
  
  /** A copy constructor.
   * @param     rO                Reference to another Animation.
   */
  Animation(const Animation& rO);
  
  /** Destructor.
   */
  virtual ~Animation();
  
  /** Assignment operator
   */
  Animation& operator = (const Animation& rO);


  ///
  /// Methods
  /// =======

  /** Adds new frame to the end of this animation.
   * @param     f                 New frame
   */
  void addFrame( const AnimFrame& f );
  
  /** Sets the index'th frame.
   * @param     index             Index of the frame being set. If bigger than
   *                              the number of frames the animation has, this
   *                              method will crash.
   * @param     f                 New frame data.
   * @exception xOutOfBounds      Thrown if index is negative or greater than
   *                              the number of frames.
   */
  void setFrame( int index,  const AnimFrame& f ) throw ( xOutOfBounds );
  
  /** Deletes the index'th frame.
   * @param     index             Index of the frame being deleted.
   * @exception xOutOfBounds      Thrown if index is negative or greater than
   *                              the number of frames.
   */
  void delFrame( int index ) throw ( xOutOfBounds );
  
  /** Deletes all the frames this animation has.
   */
  void clear();
  
  /** Sets the same frame duration for all the frames this animation has.
   * @param     fdelay            The frame delay.
   */
  void setConstantDelay( int fdelay );
  
  
  /** Sets the playmode.
   * @param     m                 New playmode.
   */
  void setPlayMode( PlayMode m );
  
  /** Jumps to the first frame.
   */
  void begin();
  
  /** Sets the play position to the index'th frame.
   * @param     index             New play position. Negative values become 0
   *                              and values greater than number of frames seek
   *                              the last frame of the animation.
   */
  void seek( int index );
  
  /** Pauses and continues the animation.
   * @param     p                 'true' pauses the animation, 'false'
   *                              continues the playing.
   */
  void pause( bool p );
  
  /** Updates the animation. If animation is paused, this won't do anything.
   * @return    'true' if framechanged happened.
   */
  bool update();


  /** Allows you to change the frames. Has the same functionality as
   * 'setFrame(...)'-method.
   * @param     index             Index of the frame.
   * @return    A reference to the index'th frame.
   * @exception xOutOfBounds      Thrown if index is negative or greater than
   *                              the number of frames this animation has.
   */
  AnimFrame& at( int index ) throw ( xOutOfBounds );
  
  /** Returns the index'th frame.
   * @param     index             Index of the frame being returned.
   * @return    The AnimFrame at given index.
   * @exception xOutOfBounds      Thrown if index is negative or greater than
   *                              the number of frames this animation has.
   */
  AnimFrame at( int index ) const throw ( xOutOfBounds );
  
  /** Loads the animation data from given file. If the loading fails, the
   * previous animation data will be lost.
   * @param     animfile          Name of the animation file.
   * @return    One of the following error codes: <code>KErrNone, KErrNotFound,
   *            KErrEof, KErrCorrupt, KErrBadIo, KErrNotSupported</code>
   */
  int load( const std::string& animfile );

  /** Reads the animation data from given stream. This method expects that
   * you have alread read the "eng2d_animation"-tag away from the stream.
   * @param     rFin              Reference to stream we read from
   * @return    One of the following error codes: <code>KErrNone, KErrEof,
   *            KErrCorrupt, KErrBadIo, KErrNotSupported</code>
   */
  int read( std::istream& rFin );
  
  /** Sets the sound mode.
   * @param     aSoundsOn         Set @c true to turn the sounds on.
   */
  void setSoundMode( bool aSoundsOn );
  

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the current play position.
   * @return      Index of the current frame.
   */
  int       getPlayPosition() const;
  
  /** Tells if this animation has been paused.
   * @return    'true' if animation has been paused.
   */
  bool      paused() const;
  
  /** Returns the current playmode.'
   * @return    Current playmode.
   */
  PlayMode  getPlayMode() const;
  
  /** Returns the data of current frame.
   * @return    The data of current frame.
   * @exception xOutOfBounds      This exception occures if there are no
   *                              frames in the animation and you're requesting
   *                              the current frame.
   */
  AnimFrame currentFrame()  const throw ( xOutOfBounds );
  
  /** Returns the number of frames this animation has.
   * @return      Number of frames.
   */
  int  frameCount() const;
  
  /** Tells if the sounds are on.
   * @return    true if sounds are on.
   */
  bool soundsOn() const;
  
  /** Returns the identifier of this animation.
   * @return  An id code of this animation.
   */
  int uid() const;
  
  /** Returns the identifier of the GfxObject that contains the graphics of
   * this animation.
   * @return  GfxObject identifier.
   */
  int gfxUid() const;

protected:

  /** Reads the HEADER element. This method expects that the opening HEADER
   * tag has been read away from the stream.
   * @param   aIn               The stream to read from.
   * @return  KErrNone if successful.
   */
  int readHeaderElement( std::istream& aIn );
  
  /** Reads the BODY element. This method expects that the opening BODY tag
   * has been read away from the stream.
   * @param   aIn               The stream to read from.
   * @return  KErrNone if successful.
   */
  int readBodyElement( std::istream& aIn );
  
   
  ///
  /// Members
  /// =======
  
  /** The playmode. */
  PlayMode      iMode;
  
  /** A counter we use when calculating frame durations. */
  int           iFrameDelayCounter;
  
  /** Current play position. */
  int           iCurrentFrame;
  
  /** Has this animation been paused. */
  bool          iPaused;
  
  /** Tells the playing direction. */
  int           iPlayDirection;
  
  /** The vector that holds the framedata. */
  std::vector<AnimFrame>  iFrameTable;
  
  /** The soundmode flag. */
  bool          iSoundsOn;
  
  /** Id of this animation. */
  int           iUid;
  
  /** Id of the graphics for this animation. */
  int           iGfxUid;
};


};


#endif
